<?php
/**
 * Alpha Tools
 *
 * @author     D-THEMES
 * @package    WP Alpha Framework
 * @subpackage Theme
 * @since      4.1.1
 */

// Direct access is denied
defined( 'ABSPATH' ) || die;

class Alpha_Version_Control extends Alpha_Base {

	/**
	 * The Page slug
	 *
	 * @since 4.1.1
	 * @access public
	 */
	public $page_slug = 'alpha-version-control';

	/**
	 * The Result
	 *
	 * @since 4.1.1
	 * @access public
	 */
	private $result;

    
	/**
	 * Theme Versions
	 *
	 * @since 4.1.1
	 * @access public
	 * @var $theme_version
	 */
	public $theme_versions = array();

	/**
	 * Plugin Versions
	 *
	 * @since 4.1.1
	 * @access public
	 * @var $plugin_versions
	 */
	public $plugin_versions = array();

	/**
	 * Theme URL
	 *
	 * @since 4.1.1
	 * @access public
	 * @var $theme_url
	 */
	public $theme_url = ALPHA_THEME_URL;

    /**
	 * Theme Slug
	 *
	 * @since 4.1.1
	 * @access public
	 * @var $theme_slug
	 */
	public $theme_slug = ALPHA_NAME;

	/**
	 * Constructor
	 *
	 * @since 4.1.1
	 * @access public
	 */
	public function __construct() {

        add_action( 'admin_menu', array( $this, 'add_admin_menu' ) );

		add_action( 'wp_ajax_alpha_modify_theme_auto_updates', array( $this, 'alpha_modify_theme_auto_updates' ) );
		add_action( 'wp_ajax_alpha_modify_plugin_auto_updates', array( $this, 'alpha_modify_plugin_auto_updates' ) );

		add_filter( 'site_transient_update_themes', array( $this, 'alpha_check_for_update_theme' ), 1 );
		add_filter( 'site_transient_update_plugins', array( $this, 'alpha_check_for_update_plugin' ), 1 );

		if ( ! current_user_can( 'administrator' ) || ! isset( $_GET['page'] ) || $this->page_slug != $_GET['page'] ) {
			return;
		}


		$this->init();
	}

    /**
	 * Initialize
	 *
	 * @since 4.1.1
	 * @access public
	 */
	public function init() {
        add_action( 'admin_enqueue_scripts', array( $this, 'enqueue' ), 30 );

		$this->theme_versions  = $this->get_theme_versions();
		$this->plugin_versions = $this->get_plugin_versions();
	}

	/**
	 * Enqueue Styles & Scripts
	 *
	 * @since 4.1.1
	 * @access public
	 */
	public function enqueue() {
		wp_enqueue_script( 'updates' );
		wp_enqueue_script( 'alpha-admin-wizard', alpha_framework_uri( '/admin/panel/wizard' . ALPHA_JS_SUFFIX ), array( 'jquery-core' ), true, 50 );

		wp_localize_script(
			'alpha-admin-wizard',
			'alpha_version_control_params',
			array(
				'wpnonce'     => wp_create_nonce( 'alpha_version_control_nonce' ),
			)
		);
	}

	/**
	 * Get Theme Versions
	 *
	 * @since 4.1.1
	 * @access public
	 *
	 * @return array versions
	 */
	public function get_theme_versions() {
		$rollback_versions = get_site_transient( 'alpha_theme_rollback_versions' );

		if ( false === $rollback_versions ) {
			$max_version   = 20;
			$current_index = 0;

			require_once ALPHA_FRAMEWORK_ADMIN . '/importer/importer-api.php';
			$importer_api = new Alpha_Importer_API();

			$versions = $importer_api->get_response( 'theme_rollback_versions' );

			if ( is_wp_error( $versions ) || empty( $versions ) ) {
				return array();
			}

			$rollback_versions = array();

			foreach ( $versions as $version ) {
				if ( $max_version <= $current_index ) {
					break;
				}

				if ( version_compare( $version, ALPHA_VERSION, '>=' ) ) {
					continue;
				}

				$current_index ++;
				$rollback_versions[] = $version;
			}

			if ( ! empty( $rollback_versions ) ) {
				set_site_transient( 'alpha_theme_rollback_versions', $rollback_versions, WEEK_IN_SECONDS );
			}
		}

		return $rollback_versions;
	}

	/**
	 * Get Plugin Versions
	 *
	 * @since 4.1.1
	 * @access public
	 *
	 * @return array plugin versions
	 */
	public function get_plugin_versions() {
		$rollback_versions = get_site_transient( 'alpha_plugin_rollback_versions' );

		if ( false === $rollback_versions ) {
			$max_version   = 20;
			$current_index = 0;

			require_once ALPHA_FRAMEWORK_ADMIN . '/importer/importer-api.php';
			$importer_api = new Alpha_Importer_API();

			$versions = $importer_api->get_response( 'plugin_rollback_versions' );

			if ( is_wp_error( $versions ) || empty( $versions ) ) {
				return array();
			}

			$rollback_versions = array();

			foreach ( $versions as $version ) {
				if ( $max_version <= $current_index ) {
					break;
				}

				if ( version_compare( $version, ALPHA_CORE_VERSION, '>=' ) ) {
					continue;
				}

				$current_index ++;
				$rollback_versions[] = $version;
			}

			set_site_transient( 'alpha_plugin_rollback_versions', $rollback_versions, WEEK_IN_SECONDS );
		}

		return $rollback_versions;
	}

	/**
	 * Check for update theme.
	 *
	 * @since 4.1.1
	 * @access public
	 *
	 * @param array $data themes info.
	 * @return array $data
	 */
	public function alpha_check_for_update_theme( $data ) {
		$transient_data = get_site_transient( 'alpha_modify_theme_auto_update' );

		if ( $transient_data ) {
			$data->response[ 'alpha' ] = $transient_data;
		}

		return $data;
	}

	/**
	 * Check for update plugin
	 *
	 * @since 4.1.1
	 * @access public
	 *
	 * @param array $data themes info.
	 * @return array $data
	 */
	public function alpha_check_for_update_plugin( $data ) {
		$transient_data = get_site_transient( 'alpha_modify_plugin_auto_update' );

		if ( ! empty( $transient_data ) ) {
			$data->response[ 'alpha-core/alpha-core.php' ] = json_decode( wp_json_encode( $transient_data ), false );
		}

		return $data;
	}

	/**
	 * Modify theme auto updates
	 *
	 * @since 4.1.1
	 * @access public
	 */
	public function alpha_modify_theme_auto_updates() {
		if ( ! isset( $_REQUEST['wpnonce'] ) || ! wp_verify_nonce( $_REQUEST['wpnonce'], 'alpha_version_control_nonce' ) ) {
			wp_send_json( false );
			die();
		}

		delete_transient( 'alpha_modify_theme_auto_update' );

		require_once ALPHA_FRAMEWORK_ADMIN . '/importer/importer-api.php';
		$importer_api = new Alpha_Importer_API();

		$args            = $importer_api->generate_args( false );
		$version         = isset( $_REQUEST['version'] ) ? wp_unslash( $_REQUEST['version'] ) : '';
		$args['version'] = $version;
		$package_url     = add_query_arg( $args, $importer_api->get_url( 'theme_rollback' ) );

		$transient_data = array(
			'theme'           => 'alpha',
			'new_version'     => $version,
			'release_version' => $version,
			'url'             => $this->theme_url,
			'package'         => $package_url,
		);

		set_site_transient( 'alpha_modify_theme_auto_update', $transient_data, WEEK_IN_SECONDS );

		add_filter( 'site_transient_update_themes', array( $this, 'alpha_check_for_update_theme' ), 1 );

		wp_send_json( true );
		die();
	}

	/**
	 * Modify plugin auto updates
	 *
	 * @since 4.1.1
	 * @access public
	 */
	public function alpha_modify_plugin_auto_updates() {
		if ( ! isset( $_REQUEST['wpnonce'] ) || ! wp_verify_nonce( $_REQUEST['wpnonce'], 'alpha_version_control_nonce' ) ) {
			wp_send_json( false );
			die();
		}

		delete_transient( 'alpha_modify_plugin_auto_update' );

		require_once ALPHA_FRAMEWORK_ADMIN . '/importer/importer-api.php';
		$importer_api = new Alpha_Importer_API();

		$args            = $importer_api->generate_args( false );
		$version         = isset( $_REQUEST['version'] ) ? wp_unslash( $_REQUEST['version'] ) : '';
		$args['version'] = $version;
		$package_url     = add_query_arg( $args, $importer_api->get_url( 'plugin_rollback' ) );

		$transient_data = array(
			'slug'            => 'alpha-core',
			'plugin'          => 'alpha-core/alpha-core.php',
			'new_version'     => $version,
			'release_version' => $version,
			'url'             => $this->theme_url,
			'package'         => $package_url,
		);

		set_site_transient( 'alpha_modify_plugin_auto_update', $transient_data, WEEK_IN_SECONDS );

		add_filter( 'site_transient_update_plugins', array( $this, 'alpha_check_for_update_plugin' ), 1 );

		wp_send_json( true );
		die();
	}


	/**
	 * Add Tools to admin menu
	 *
	 * @since 1.0
	 * @access public
	 */
	public function add_admin_menu() {
		add_submenu_page( 'alpha', esc_html__( 'Version Control', 'alpha' ), esc_html__( 'Version Control', 'alpha' ), 'manage_options', $this->page_slug, array( $this, 'view_tools' ), 3 );
	}

	/**
	 * Render tools page
	 *
	 * @since 1.0
	 * @access public
	 */
	public function view_tools() {
        $admin_config = Alpha_Admin::get_instance()->admin_config;
		$title        = array(
			'title' => esc_html__( 'Version Control', 'alpha' ),
			'desc'  => sprintf( esc_html__( 'Experiencing an issue with New version? Rollback to a previous version before the issue appeared.', 'alpha' ), ALPHA_DISPLAY_NAME ),
		);
		Alpha_Admin_Panel::get_instance()->view_header( 'version-control', $admin_config, $title );

		$nonce = wp_create_nonce( 'alpha-version-control' );
		?>
		<?php
		if ( isset( $this->result ) ) {
			if ( $this->result['success'] ) {
				echo '<div class="alpha-notify updated inline"><p>' . esc_html( $this->result['message'] ) . '</p></div>';
			} else {
				echo '<div class="alpha-notify error inline"><p>' . esc_html( $this->result['message'] ) . '</p></div>';
			}
		}
		?>
		<div class="alpha-admin-panel-body alpha-card-box alpha-version-control">
			<table class="wp-list-table widefat" id="alpha_versions_table">
				<thead>
					<tr>
						<th scope="col" id="title" class="manage-column column-title column-primary"><?php esc_html_e( 'Action Name', 'alpha' ); ?></th>
						<th scope="col" id="remove" class="manage-column column-remove"><?php esc_html_e( 'Action', 'alpha' ); ?></th>
					</tr>
				</thead>
				<tbody id="the-list">
					<tr class="theme-version" id="alpha-theme-version">
						<th>
							<strong class="action-name"><?php echo esc_html( ALPHA_DISPLAY_NAME ); ?></strong>
							<p class="description warning"><?php echo esc_html__( 'Warning: Please backup your database before making the rollback.', 'alpha' ); ?></p>
						</th>
						<td class="run-tool">
							<select class="version-select theme-versions" id="theme-versions">
								<?php
								foreach ( $this->theme_versions as $version ) {
									?>
										<option value="<?php echo esc_attr( $version ); ?>"><?php echo esc_html( $version ); ?></option>
										<?php
								}
								?>
							</select>
							<a href="#" class="button button-large button-light theme-rollback"><?php echo esc_html__( 'Downgrade', 'alpha' ); ?></a>
						</td>
					</tr>
					<tr class="plugin-version" id="alpha-plugin-version">
						<th>
							<strong class="action-name"><?php echo esc_html( ALPHA_DISPLAY_NAME . ' Core Plugin' ); ?></strong>
							<p class="description warning"><?php echo esc_html__( 'Warning: Please backup your database before making the rollback.', 'alpha' ); ?></p>
						</th>
						<td class="run-tool">
							<select class="version-select plugin-versions" id="plugin-versions">
								<?php
								foreach ( $this->plugin_versions as $version ) {
									?>
										<option value="<?php echo esc_attr( $version ); ?>"><?php echo esc_html( $version ); ?></option>
										<?php
								}
								?>
							</select>
							<a href="#" class="button button-large button-light plugin-rollback"><?php echo esc_html__( 'Downgrade', 'alpha' ); ?></a>
						</td>
					</tr>
				</tbody>
			</table>
		</div>
		<?php
		Alpha_Admin_Panel::get_instance()->view_footer();
	}
}

Alpha_Version_Control::get_instance();
