<?php
/**
 * Alpha Icon List Widget
 *
 * Alpha Widget to display icon list.
 *
 * @author     D-THEMES
 * @package    WP Alpha Core FrameWork
 * @subpackage Core
 * @since      1.0
 */

defined( 'ABSPATH' ) || die;

use Elementor\Controls_Manager;
use Elementor\Repeater;
use Elementor\Group_Control_Typography;
use Elementor\Group_Control_Text_Shadow;
use ELementor\Group_Control_Box_Shadow;

class Alpha_IconList_Elementor_Widget extends \Elementor\Widget_Base {

	public function get_name() {
		return ALPHA_NAME . '_widget_iconlist';
	}

	public function get_title() {
		return esc_html__( 'Icon List', 'alpha-core' );
	}

	public function get_categories() {
		return array( 'alpha_widget' );
	}

	public function get_keywords() {
		return array( 'iconlist', 'alpha', 'menu' );
	}

	public function get_icon() {
		return 'alpha-elementor-widget-icon eicon-bullet-list';
	}

	/**
	 * Get the style depends.
	 *
	 * @since 1.2.0
	 */
	public function get_style_depends() {
		wp_register_style( 'alpha-icon-list', alpha_core_framework_uri( '/widgets/iconlist/iconlist' . ( is_rtl() ? '-rtl' : '' ) . '.min.css' ), array(), ALPHA_CORE_VERSION );
		return array( 'alpha-icon-list' );
	}

	/**
	 * Register icon list widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0
	 * @access protected
	 */
	protected function register_controls() {

		$left   = is_rtl() ? 'right' : 'left';
		$right  = 'left' == $left ? 'right' : 'left';
		$before = is_rtl() ? 'after' : 'before';
		$after  = 'before' == $before ? 'after' : 'before';

		$this->start_controls_section(
			'section_icon',
			array(
				'label' => esc_html__( 'Icon List', 'alpha-core' ),
			)
		);

		$this->add_control(
			'view',
			array(
				'label'       => esc_html__( 'Layout', 'alpha-core' ),
				'type'        => Controls_Manager::CHOOSE,
				'default'     => 'block',
				'options'     => array(
					'block'  => array(
						'title' => esc_html__( 'Default', 'alpha-core' ),
						'icon'  => 'eicon-editor-list-ul',
					),
					'inline' => array(
						'title' => esc_html__( 'Inline', 'alpha-core' ),
						'icon'  => 'eicon-ellipsis-h',
					),
				),
				'description' => esc_html__( 'Pick out your iconlist layout.', 'alpha-core' ),
				'toggle'      => false,
			)
		);

		$this->add_responsive_control(
			'icon_h_align',
			array(
				'label'       => esc_html__( 'Alignment', 'alpha-core' ),
				'description' => esc_html__( 'Control the horizontal alignment of iconlists.', 'alpha-core' ),
				'type'        => Controls_Manager::CHOOSE,
				'default'     => 'flex-start',
				'toggle'      => false,
				'options'     => array(
					'flex-start' => array(
						'title' => esc_html__( 'Left', 'alpha-core' ),
						'icon'  => "eicon-h-align-{$left}",
					),
					'center'     => array(
						'title' => esc_html__( 'Center', 'alpha-core' ),
						'icon'  => 'eicon-h-align-center',
					),
					'flex-end'   => array(
						'title' => esc_html__( 'Right', 'alpha-core' ),
						'icon'  => "eicon-h-align-{$right}",
					),
				),
				'selectors'   => array(
					'.elementor-element-{{ID}} .alpha-icon-lists' => 'align-items: {{VALUE}};',
				),
				'condition'   => array(
					'view' => 'block',
				),
			)
		);

		$this->add_responsive_control(
			'icon_v_align',
			array(
				'label'       => esc_html__( 'Alignment', 'alpha-core' ),
				'description' => esc_html__( 'Control the horizontal alignments of iconlists.', 'alpha-core' ),
				'type'        => Controls_Manager::CHOOSE,
				'default'     => 'center',
				'options'     => array(
					'flex-start' => array(
						'title' => esc_html__( 'Left', 'alpha-core' ),
						'icon'  => 'eicon-h-align-left',
					),
					'center'     => array(
						'title' => esc_html__( 'Center', 'alpha-core' ),
						'icon'  => 'eicon-h-align-center',
					),
					'flex-end'   => array(
						'title' => esc_html__( 'Right', 'alpha-core' ),
						'icon'  => 'eicon-h-align-right',
					),
				),
				'selectors'   => array(
					'.elementor-element-{{ID}} .alpha-icon-lists' => 'justify-content: {{VALUE}};',
				),
				'condition'   => array(
					'view' => 'inline',
				),
			)
		);

		$this->add_control(
			'title',
			array(
				'label'       => esc_html__( 'Title', 'alpha-core' ),
				'dynamic'     => array(
					'active' => true,
				),
				'type'        => Controls_Manager::TEXTAREA,
				'placeholder' => esc_html__( 'Enter your title', 'alpha-core' ),
				'description' => esc_html__( 'Enter your iconlist title.', 'alpha-core' ),
			)
		);

		$repeater = new Repeater();

		$repeater->add_control(
			'text',
			array(
				'label'       => esc_html__( 'Text', 'alpha-core' ),
				'type'        => Controls_Manager::TEXT,
				'label_block' => true,
				'placeholder' => esc_html__( 'List Item', 'alpha-core' ),
				'default'     => esc_html__( 'List Item', 'alpha-core' ),
				'description' => esc_html__( 'Enter iconlist item text.', 'alpha-core' ),
				'dynamic'     => array(
					'active' => true,
				),
			)
		);

		$repeater->add_control(
			'selected_icon',
			array(
				'label'       => esc_html__( 'Icon', 'alpha-core' ),
				'type'        => Controls_Manager::ICONS,
				'description' => esc_html__( 'Choose any one from our icon library.', 'alpha-core' ),
				'default'     => array(
					'value'   => 'fas fa-check',
					'library' => 'fa-solid',
				),
			)
		);

		$repeater->add_control(
			'link',
			array(
				'label'       => esc_html__( 'Link', 'alpha-core' ),
				'type'        => Controls_Manager::URL,
				'description' => esc_html__( 'Input URL where you are able to access when you click icon list item.', 'alpha-core' ),
				'dynamic'     => array(
					'active' => true,
				),
				'placeholder' => 'https://your-link.com',
			)
		);

		$this->add_control(
			'icon_list',
			array(
				'label'       => esc_html__( 'Items', 'alpha-core' ),
				'type'        => Controls_Manager::REPEATER,
				'fields'      => $repeater->get_controls(),
				'default'     => array(
					array(
						'text'          => esc_html__( 'List Item #1', 'alpha-core' ),
						'selected_icon' => array(
							'value'   => 'fas fa-check',
							'library' => 'fa-solid',
						),
					),
					array(
						'text'          => esc_html__( 'List Item #2', 'alpha-core' ),
						'selected_icon' => array(
							'value'   => 'fas fa-times',
							'library' => 'fa-solid',
						),
					),
					array(
						'text'          => esc_html__( 'List Item #3', 'alpha-core' ),
						'selected_icon' => array(
							'value'   => 'fas fa-dot-circle',
							'library' => 'fa-solid',
						),
					),
				),
				'title_field' => '{{{ elementor.helpers.renderIcon( this, selected_icon, {}, "i", "panel" ) || \'<i class="{{ icon }}" aria-hidden="true"></i>\' }}} {{{ text }}}',
			)
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_title',
			array(
				'label'     => esc_html__( 'Title', 'alpha-core' ),
				'tab'       => Controls_Manager::TAB_STYLE,
				'condition' => array(
					'title!' => '',
				),
			)
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'title_typography',
				'selector' => '.elementor-element-{{ID}} .list-title',
			)
		);

		$this->add_control(
			'title_color',
			array(
				'label'       => esc_html__( 'Color', 'alpha-core' ),
				'type'        => Controls_Manager::COLOR,
				'description' => esc_html__( 'Set title color', 'alpha-core' ),
				'default'     => '',
				'selectors'   => array(
					'.elementor-element-{{ID}} .list-title' => 'color: {{VALUE}}',
				),
			)
		);

		$this->add_control(
			'title_spacing',
			array(
				'label'       => esc_html__( 'Spacing', 'alpha-core' ),
				'type'        => Controls_Manager::SLIDER,
				'range'       => array(
					'px'  => array(
						'step' => 1,
						'min'  => 0,
						'max'  => 100,
					),
					'rem' => array(
						'step' => 0.5,
						'min'  => 1,
						'max'  => 20,
					),
				),
				'description' => esc_html__( 'Controls the gap space between title and content.', 'alpha-core' ),
				'size_units'  => array( 'px', 'rem' ),
				'selectors'   => array(
					'.elementor-element-{{ID}} .block-type .list-title' => 'margin-bottom: {{SIZE}}{{UNIT}};',
					'.elementor-element-{{ID}} .inline-type .list-title' => "margin-{$right}: {{SIZE}}{{UNIT}};",
				),
			)
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_icon_list',
			array(
				'label' => esc_html__( 'List Item', 'alpha-core' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'item_typography',
				'selector' => '.elementor-element-{{ID}} .alpha-icon-lists .alpha-icon-list-item',
			)
		);

		$this->add_responsive_control(
			'item_padding',
			array(
				'label'       => esc_html__( 'Padding', 'alpha-core' ),
				'type'        => Controls_Manager::DIMENSIONS,
				'description' => esc_html__( 'Controls the padding dimensions of iconlist item.', 'alpha-core' ),
				'size_units'  => array( 'px', 'rem' ),
				'default'     => array(
					'top'    => 5,
					'right'  => 10,
					'bottom' => 5,
					'left'   => 10,
				),
				'selectors'   => array(
					'.elementor-element-{{ID}} .alpha-icon-lists .alpha-icon-list-item' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				),
			)
		);

		$this->add_control(
			'text_color',
			array(
				'label'       => esc_html__( 'Color', 'alpha-core' ),
				'type'        => Controls_Manager::COLOR,
				'description' => esc_html__( 'Controls the color of list item.', 'alpha-core' ),
				'default'     => '',
				'selectors'   => array(
					'.elementor-element-{{ID}} .alpha-icon-lists .alpha-icon-list-item' => 'color: {{VALUE}}',
					'.elementor-element-{{ID}} .alpha-icon-lists .alpha-icon-list-item svg' => 'fill: {{VALUE}}',
				),
			)
		);

		$this->add_control(
			'text_color_hover',
			array(
				'label'       => esc_html__( 'Hover Color', 'alpha-core' ),
				'description' => esc_html__( 'Controls the hover color of list item.', 'alpha-core' ),
				'type'        => Controls_Manager::COLOR,
				'default'     => '',
				'selectors'   => array(
					'.elementor-element-{{ID}} .alpha-icon-lists .alpha-icon-list-item:hover' => 'color: {{VALUE}}',
					'.elementor-element-{{ID}} .alpha-icon-lists .alpha-icon-list-item:hover i' => 'color: {{VALUE}}',
					'.elementor-element-{{ID}} .alpha-icon-lists .alpha-icon-list-item:hover svg' => 'fill: {{VALUE}}',
				),
			)
		);

		$this->add_control(
			'divider',
			array(
				'label'       => esc_html__( 'Divider', 'alpha-core' ),
				'description' => esc_html__( 'Enables to show divider between list items.', 'alpha-core' ),
				'type'        => Controls_Manager::SWITCHER,
				'label_off'   => esc_html__( 'Off', 'alpha-core' ),
				'label_on'    => esc_html__( 'On', 'alpha-core' ),
				'selectors'   => array(
					".elementor-element-{{ID}} .alpha-icon-lists .alpha-icon-list-item:not(:last-child):{$after}" => 'content:""',
				),
				'separator'   => 'before',
			)
		);

		$this->add_control(
			'divider_style',
			array(
				'label'       => esc_html__( 'Style', 'alpha-core' ),
				'description' => esc_html__( 'Selects proper divider style from 4 default types.', 'alpha-core' ),
				'type'        => Controls_Manager::SELECT,
				'options'     => array(
					'solid'  => esc_html__( 'Solid', 'alpha-core' ),
					'double' => esc_html__( 'Double', 'alpha-core' ),
					'dotted' => esc_html__( 'Dotted', 'alpha-core' ),
					'dashed' => esc_html__( 'Dashed', 'alpha-core' ),
				),
				'default'     => 'solid',
				'condition'   => array(
					'divider' => 'yes',
				),
				'selectors'   => array(
					".elementor-element-{{ID}} .alpha-icon-lists .alpha-icon-list-item:not(:last-child):{$after}" => 'border-style: {{VALUE}}',
				),
			)
		);

		$this->add_control(
			'divider_block_width',
			array(
				'label'       => esc_html__( 'Width', 'alpha - core' ),
				'type'        => Controls_Manager::SLIDER,
				'size_units'  => array( '%', 'px' ),
				'description' => esc_html__( 'Determines divider width between list items.', 'alpha-core' ),
				'default'     => array(
					'size' => 1,
					'unit' => 'px',
				),
				'range'       => array(
					'%' => array(
						'min' => 1,
						'max' => 100,
					),
				),
				'condition'   => array(
					'divider' => 'yes',
					'view'    => 'block',
				),
				'selectors'   => array(
					".elementor-element-{{ID}} .alpha-icon-lists.block-type .alpha-icon-list-item:not(:last-child):{$after}" => 'width: {{SIZE}}{{UNIT}}',
				),
			)
		);

		$this->add_control(
			'divider_inline_width',
			array(
				'label'       => esc_html__( 'Width (px)', 'alpha - core' ),
				'type'        => Controls_Manager::SLIDER,
				'size_units'  => array( 'px' ),
				'description' => esc_html__( 'Determines divider width between list items.', 'alpha-core' ),
				'default'     => array(
					'size' => 1,
					'unit' => 'px',
				),
				'range'       => array(
					'px' => array(
						'min' => 1,
						'max' => 100,
					),
				),
				'condition'   => array(
					'divider' => 'yes',
					'view'    => 'inline',
				),
				'selectors'   => array(
					".elementor-element-{{ID}} .alpha-icon-lists.inline-type .alpha-icon-list-item:not(:last-child):{$after}" => "border-{$right}-width: {{SIZE}}{{UNIT}};",
					'.elementor-element-{{ID}} .alpha-icon-lists.inline-type .alpha-icon-list-item:not(:last-child)' => "margin-{$right}: {{SIZE}}{{UNIT}}",
				),
			)
		);

		$this->add_control(
			'divider_block_height',
			array(
				'label'       => esc_html__( 'Height (px)', 'alpha-core' ),
				'type'        => Controls_Manager::SLIDER,
				'size_units'  => array( 'px' ),
				'description' => esc_html__( 'Determines divider height between list items.', 'alpha-core' ),
				'default'     => array(
					'size' => 14,
					'unit' => 'px',
				),
				'range'       => array(
					'px' => array(
						'min' => 1,
						'max' => 100,
					),
				),
				'condition'   => array(
					'divider' => 'yes',
					'view'    => 'block',
				),
				'selectors'   => array(
					".elementor-element-{{ID}} .alpha-icon-lists.block-type .alpha-icon-list-item:not(:last-child):{$after}" => 'border-bottom-width: {{SIZE}}{{UNIT}};',
					'.elementor-element-{{ID}} .alpha-icon-lists.block-type .alpha-icon-list-item:not(:last-child)' => 'margin-bottom: {{SIZE}}{{UNIT}}',
				),
			)
		);

		$this->add_control(
			'divider_inline_height',
			array(
				'label'       => esc_html__( 'Height', 'alpha-core' ),
				'type'        => Controls_Manager::SLIDER,
				'size_units'  => array( '%', 'px' ),
				'description' => esc_html__( 'Determines divider height between list items.', 'alpha-core' ),
				'default'     => array(
					'size' => 14,
					'unit' => 'px',
				),
				'range'       => array(
					'%' => array(
						'min' => 1,
						'max' => 100,
					),
				),
				'condition'   => array(
					'divider' => 'yes',
					'view'    => 'inline',
				),
				'selectors'   => array(
					".elementor-element-{{ID}} .alpha-icon-lists.inline-type .alpha-icon-list-item:not(:last-child):{$after}" => 'height: {{SIZE}}{{UNIT}}',
				),
			)
		);

		$this->add_control(
			'divider_color',
			array(
				'label'       => esc_html__( 'Color', 'alpha-core' ),
				'type'        => Controls_Manager::COLOR,
				'default'     => '#ddd',
				'description' => esc_html__( 'Controls the divider color.', 'alpha-core' ),
				'condition'   => array(
					'divider' => 'yes',
				),
				'selectors'   => array(
					".elementor-element-{{ID}} .alpha-icon-lists .alpha-icon-list-item:not(:last-child):{$after}" => 'border-color: {{VALUE}}',
				),
			)
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_icon_style',
			array(
				'label' => esc_html__( 'Icon', 'alpha-core' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_control(
			'icon_size',
			array(
				'label'       => esc_html__( 'Size', 'alpha-core' ),
				'type'        => Controls_Manager::SLIDER,
				'description' => esc_html__( 'Set the icon size of list item.', 'alpha-core' ),
				'default'     => array(
					'size' => 14,
				),
				'range'       => array(
					'px' => array(
						'min' => 6,
					),
				),
				'selectors'   => array(
					'.elementor-element-{{ID}} .alpha-icon-lists .alpha-icon-list-item i' => 'font-size: {{SIZE}}{{UNIT}}',
					'.elementor-element-{{ID}} .alpha-icon-lists .alpha-icon-list-item svg' => 'width: {{SIZE}}{{UNIT}}',
				),
			)
		);

		$this->add_control(
			'icon_spacing',
			array(
				'label'       => esc_html__( 'Spacing', 'alpha-core' ),
				'type'        => Controls_Manager::SLIDER,
				'description' => esc_html__( 'Controls the gap space between icon and text.', 'alpha-core' ),
				'size_units'  => array( 'px', 'rem' ),
				'range'       => array(
					'px'  => array(
						'step' => 1,
						'min'  => 0,
						'max'  => 100,
					),
					'rem' => array(
						'step' => 0.5,
						'min'  => 0,
						'max'  => 20,
					),
				),
				'default'     => array(
					'size' => 5,
					'unit' => 'px',
				),
				'selectors'   => array(
					'.elementor-element-{{ID}} .alpha-icon-lists .alpha-icon-list-item i' => "padding-{$right}: {{SIZE}}{{UNIT}};",
					'.elementor-element-{{ID}} .alpha-icon-lists .alpha-icon-list-item svg' => "margin-{$right}: {{SIZE}}{{UNIT}};",
				),
			)
		);

		$this->add_control(
			'icon_color',
			array(
				'label'       => esc_html__( 'Color', 'alpha-core' ),
				'description' => esc_html__( 'Set the color of list item icon.', 'alpha-core' ),
				'type'        => Controls_Manager::COLOR,
				'default'     => '',
				'separator'   => 'before',
				'selectors'   => array(
					'.elementor-element-{{ID}} .alpha-icon-lists .alpha-icon-list-item i' => 'color: {{VALUE}}',
					'.elementor-element-{{ID}} .alpha-icon-lists .alpha-icon-list-item svg' => 'fill: {{VALUE}}',
				),
			)
		);

		$this->add_control(
			'icon_color_hover',
			array(
				'label'       => esc_html__( 'Hover Color', 'alpha-core' ),
				'description' => esc_html__( 'Set the hover color of list item icon.', 'alpha-core' ),
				'type'        => Controls_Manager::COLOR,
				'default'     => '',
				'selectors'   => array(
					'.elementor-element-{{ID}} .alpha-icon-lists .alpha-icon-list-item i:hover' => 'color: {{VALUE}}',
					'.elementor-element-{{ID}} .alpha-icon-lists .alpha-icon-list-item svg:hover' => 'fill: {{VALUE}}',
				),
			)
		);

		$this->end_controls_section();
	}

	protected function render() {
		$atts = $this->get_settings_for_display();
		require alpha_core_framework_path( ALPHA_CORE_FRAMEWORK_PATH . '/widgets/iconlist/render-iconlist-elementor.php' );
	}
}
