<?php
/**
 * Alpha Highlight Widget
 *
 * Alpha Widget to display Highlight text.
 *
 * @author     D-THEMES
 * @package    WP Alpha Core FrameWork
 * @subpackage Core
 * @since      1.0.0
 */

defined( 'ABSPATH' ) || die;

use Elementor\Repeater;
use Elementor\Controls_Manager;
use Elementor\Core\Schemes\Color;
use Elementor\Group_Control_Typography;
use Elementor\Core\Schemes\Typography;

class Alpha_Highlight_Elementor_Widget extends Elementor\Widget_Heading {

	public function get_name() {
		return ALPHA_NAME . '_widget_highlight';
	}

	public function get_title() {
		return esc_html__( 'Highlight', 'alpha-core' );
	}

	public function get_categories() {
		return array( 'alpha_widget' );
	}

	public function get_icon() {
		return 'alpha-elementor-widget-icon alpha-widget-icon-highlight';
	}

	public function get_keywords() {
		return array( 'highlight', 'animated', 'heading', 'text', 'alpha' );
	}

	/**
	 * Get the style depends.
	 *
	 * @since 1.2.0
	 */
	public function get_style_depends() {
		wp_register_style( 'alpha-highlight', alpha_core_framework_uri( '/widgets/highlight/highlight' . ( is_rtl() ? '-rtl' : '' ) . '.min.css' ), array(), ALPHA_CORE_VERSION );
		return array( 'alpha-highlight' );
	}

	public function get_script_depends() {
		return array();
	}

	protected function register_controls() {

		parent::register_controls();

		$repeater = new Repeater();

		$repeater->add_control(
			'text',
			array(
				'label'       => esc_html__( 'Text', 'alpha-core' ),
				'description' => esc_html__( 'Input the text.', 'alpha-core' ),
				'label_block' => true,
				'type'        => Controls_Manager::TEXT,
				'default'     => '',
				'dynamic'     => array(
					'active' => true,
				),
			)
		);

		$repeater->add_control(
			'highlight',
			array(
				'label'       => esc_html__( 'Highlight', 'alpha-core' ),
				'description' => esc_html__( 'Make the text highlight one.', 'alpha-core' ),
				'type'        => Controls_Manager::SWITCHER,
				'label_on'    => esc_html__( 'Yes', 'alpha-core' ),
				'label_off'   => esc_html__( 'No', 'alpha-core' ),
				'default'     => false,
			)
		);

		$repeater->add_control(
			'text_color',
			array(
				'label'       => esc_html__( 'Text Color', 'alpha-core' ),
				'description' => esc_html__( 'Set the text color.', 'alpha-core' ),
				'type'        => Controls_Manager::COLOR,
				'selectors'   => array(
					'.elementor-element-{{ID}} {{CURRENT_ITEM}}' => 'color: {{VALUE}}',
				),
				'condition'   => array(
					'highlight' => 'yes',
				),
			)
		);

		$repeater->add_control(
			'text_color_hover',
			array(
				'label'       => esc_html__( 'Text Hover Color', 'alpha-core' ),
				'description' => esc_html__( 'Set the text hover color.', 'alpha-core' ),
				'type'        => Controls_Manager::COLOR,
				'selectors'   => array(
					'.elementor-element-{{ID}} {{CURRENT_ITEM}}:hover' => 'color: {{VALUE}}',
				),
				'condition'   => array(
					'highlight' => 'yes',
				),
			)
		);

		$repeater->add_control(
			'highlight_color',
			array(
				'label'       => esc_html__( 'Highlight Color', 'alpha-core' ),
				'description' => esc_html__( 'Determine the text highlight color.', 'alpha-core' ),
				'type'        => Controls_Manager::COLOR,
				'default'     => alpha_get_option( 'primary_color' ),
				'selectors'   => array(
					'.elementor-element-{{ID}} {{CURRENT_ITEM}}' => 'background-image: linear-gradient({{VALUE}}, {{VALUE}})',
				),
				'condition'   => array(
					'highlight' => 'yes',
				),
			)
		);

		$repeater->add_control(
			'highlight_height',
			array(
				'label'       => esc_html__( 'Height (%)', 'alpha-core' ),
				'type'        => Controls_Manager::SLIDER,
				'size_units'  => array( '%' ),
				'range'       => array(
					'%' => array(
						'step' => 1,
						'min'  => 0,
						'max'  => 100,
					),
				),
				'selectors'   => array(
					// '.elementor-element-{{ID}} {{CURRENT_ITEM}}' => 'background-size: 0 {{SIZE}}%',
					// '.elementor-element-{{ID}} {{CURRENT_ITEM}}.animating' => 'background-size: 100% {{SIZE}}%',
					'.elementor-element-{{ID}} {{CURRENT_ITEM}}' => '--alpha-highlight-bg-size: {{SIZE}}%',
				),
				'description' => esc_html__( 'Determine the highlight part within a text.', 'alpha-core' ),
				'condition'   => array(
					'highlight' => 'yes',
				),
			)
		);

		$repeater->add_control(
			'line_break',
			array(
				'label'       => esc_html__( 'Line Break', 'alpha-core' ),
				'description' => esc_html__( 'It will be placed on a whole line.', 'alpha-core' ),
				'type'        => Controls_Manager::SWITCHER,
			)
		);

		$repeater->add_control(
			'custom_class',
			array(
				'label'       => esc_html__( 'Custom Class', 'alpha-core' ),
				'description' => esc_html__( 'Add some custom classes.', 'alpha-core' ),
				'type'        => Controls_Manager::TEXT,
			)
		);

		$presets = array(
			array(
				'text' => esc_html__( 'Have your', 'alpha-core' ),
			),
			array(
				'text'      => esc_html__( 'website popup', 'alpha-core' ),
				'highlight' => 'yes',
			),
			array(
				'text' => esc_html__( 'out with', 'alpha-core' ),
			),
			array(
				'text'      => esc_html__( 'Our Theme', 'alpha-core' ),
				'highlight' => 'yes',
			),
		);

		$this->add_control(
			'items',
			array(
				'label'       => esc_html__( 'Content', 'alpha-core' ),
				'type'        => Controls_Manager::REPEATER,
				'title_field' => '{{{ text }}}',
				'fields'      => $repeater->get_controls(),
				'default'     => $presets,
				'description' => esc_html__( 'Add repeater items.', 'alpha-core' ),
			),
			array(
				'position' => array(
					'at' => 'after',
					'of' => 'title',
				),
			)
		);

		$this->remove_control( 'title' );
		$this->remove_control( 'link' );
		$this->remove_control( 'size' );
	}

	protected function render() {
		$atts         = $this->get_settings_for_display();
		$atts['self'] = $this;
		require alpha_core_framework_path( ALPHA_CORE_FRAMEWORK_PATH . '/widgets/highlight/render-highlight-elementor.php' );
	}

	protected function content_template() {
		?>
		<#
		var headerSizeTag = elementor.helpers.validateHTMLTag( settings.header_size );
		view.addRenderAttribute( 'title', 'class', ['elementor-heading-title', 'highlight-text'] );
		#>
		<{{{ headerSizeTag }}} {{{ view.getRenderAttributeString( 'title' ) }}}>
			<#
			_.each( settings.items, function( item, index ) {
				let item_key = view.getRepeaterSettingKey( 'text', 'items', index );
				view.addRenderAttribute(item_key, 'class', item.custom_class);
				
				if(item._id && item.highlight) {
					view.addRenderAttribute( item_key, 'class', 'highlight animating elementor-repeater-item-' + item._id );
					view.addInlineEditingAttributes( item_key );
				}

				var lineBreak = '';
				if ('yes' == item.line_break) {
					lineBreak = '<br>';
				}
				#>
				<span {{{ view.getRenderAttributeString( item_key ) }}}>{{{ item.text }}}{{{ lineBreak }}}</span>
				<#
			});
			#>
		</{{{ headerSizeTag }}}>
		<?php
	}
}
