<?php
/**
 * Alpha Filter Widget
 *
 * Alpha Widget to display filter for products.
 *
 * @author     D-THEMES
 * @package    WP Alpha Core FrameWork
 * @subpackage Core
 * @since      1.0.0
 */

defined( 'ABSPATH' ) || die;

use Elementor\Controls_Manager;
use Elementor\Core\Schemes\Color;
use Elementor\Group_Control_Typography;
use Elementor\Core\Schemes\Typography;
use Elementor\Repeater;
use Elementor\Group_Control_Border;
use Elementor\Alpha_Controls_Manager;

class Alpha_Filter_Elementor_Widget extends \Elementor\Widget_Base {
	public $attributes;

	public function get_name() {
		return ALPHA_NAME . '_widget_filter';
	}

	public function get_title() {
		return esc_html__( 'Filter', 'alpha-core' );
	}

	public function get_icon() {
		return 'alpha-elementor-widget-icon alpha-widget-icon-filter';
	}

	public function get_categories() {
		return array( 'alpha_widget' );
	}

	public function get_keywords() {
		return array( 'filter', 'product', 'attribute', 'category', 'tag' );
	}

	/**
	 * Get the style depends.
	 *
	 * @since 1.2.0
	 */
	public function get_style_depends() {
		wp_register_style( 'alpha-filter', alpha_core_framework_uri( '/widgets/filter/filter' . ( is_rtl() ? '-rtl' : '' ) . '.min.css' ), array(), ALPHA_CORE_VERSION );
		return array( 'alpha-filter' );
	}

	public function get_script_depends() {
		return array( 'alpha-shop' );
	}

	protected function register_controls() {

		$left  = is_rtl() ? 'right' : 'left';
		$right = 'left' == $left ? 'right' : 'left';

		$this->start_controls_section(
			'section_filter_content',
			array(
				'label' => esc_html__( 'Filter', 'alpha-core' ),
			)
		);

			$this->attributes = array();
			$taxonomies       = wc_get_attribute_taxonomies();
			$default_att      = '';

		if ( ! count( $taxonomies ) ) {
			$this->add_control(
				'sorry_desc',
				array(
					'description' => sprintf( esc_html__( 'Sorry, there are no product attributes available in this site. Click %1$shere%2$s to add attributes.', 'alpha-core' ), '<a href="' . admin_url() . 'edit.php?post_type=product&page=product_attributes" target="blank">', '</a>' ),
					'type'        => Alpha_Controls_Manager::DESCRIPTION,
				)
			);

			$this->end_controls_section();

			return;
		}

		foreach ( $taxonomies as $key => $value ) {
			$this->attributes[ 'pa_' . $value->attribute_name ] = $value->attribute_label;
			if ( ! $default_att ) {
				$default_att = 'pa_' . $value->attribute_name;
			}
		}

			$repeater = new Repeater();

			$repeater->add_control(
				'name',
				array(
					'label'       => esc_html__( 'Attribute', 'alpha-core' ),
					'description' => esc_html__( 'Select product attribute you want.', 'alpha-core' ),
					'type'        => Controls_Manager::SELECT,
					'options'     => $this->attributes,
					'default'     => $default_att,
				)
			);

			$repeater->add_control(
				'query_opt',
				array(
					'label'       => esc_html__( 'Query Type', 'alpha-core' ),
					'description' => esc_html__( 'Select query type.', 'alpha-core' ),
					'type'        => Controls_Manager::SELECT,
					'options'     => array(
						'and' => esc_html__( 'AND', 'alpha-core' ),
						'or'  => esc_html__( 'OR', 'alpha-core' ),
					),
					'default'     => 'or',
				)
			);

			$this->add_control(
				'attributes',
				array(
					'label'       => esc_html__( 'Product Attributes', 'alpha-core' ),
					'type'        => Controls_Manager::REPEATER,
					'fields'      => $repeater->get_controls(),
					'default'     => array(
						array(
							'name'      => $default_att,
							'query_opt' => 'or',
						),
					),
					'title_field' => '{{{ name }}}',
				)
			);

			$this->add_responsive_control(
				'align',
				array(
					'label'       => esc_html__( 'Align', 'alpha-core' ),
					'type'        => Controls_Manager::CHOOSE,
					'description' => esc_html__( 'Control the alignment of filter section.', 'alpha-core' ),
					'default'     => 'center',
					'options'     => array(
						'left'   => array(
							'title' => esc_html__( 'Left', 'alpha-core' ),
							'icon'  => 'eicon-text-align-left',
						),
						'center' => array(
							'title' => esc_html__( 'Center', 'alpha-core' ),
							'icon'  => 'eicon-text-align-center',
						),
						'right'  => array(
							'title' => esc_html__( 'Right', 'alpha-core' ),
							'icon'  => 'eicon-text-align-right',
						),
					),
				)
			);

			$this->add_control(
				'filter_button_heading',
				array(
					'label'     => esc_html__( 'Filter Button', 'alpha-core' ),
					'type'      => Controls_Manager::HEADING,
					'separator' => 'before',
				)
			);

			$this->add_control(
				'btn_label',
				array(
					'type'        => Controls_Manager::TEXT,
					'label'       => esc_html__( 'Label', 'alpha-core' ),
					'description' => esc_html__( 'Input filter button label.', 'alpha-core' ),
					'default'     => 'Filter',
				)
			);

			$this->add_control(
				'btn_skin',
				array(
					'type'        => Controls_Manager::SELECT,
					'label'       => esc_html__( 'Skin', 'alpha-core' ),
					'description' => esc_html__( 'Select filter button skin.', 'alpha-core' ),
					'options'     => array(
						'btn-primary'   => esc_html__( 'Primary', 'alpha-core' ),
						'btn-secondary' => esc_html__( 'Secondary', 'alpha-core' ),
						'btn-warning'   => esc_html__( 'Warning', 'alpha-core' ),
						'btn-danger'    => esc_html__( 'Danger', 'alpha-core' ),
						'btn-success'   => esc_html__( 'Success', 'alpha-core' ),
						'btn-dark'      => esc_html__( 'Dark', 'alpha-core' ),
						'btn-white'     => esc_html__( 'White', 'alpha-core' ),
					),
					'default'     => 'btn-primary',
				)
			);

			$this->add_control(
				'btn_size',
				array(
					'label'       => esc_html__( 'Size', 'alpha-core' ),
					'description' => esc_html__( 'Choose button size. Choose from Small, Medium, Normal, Large.', 'alpha-core' ),
					'type'        => Controls_Manager::CHOOSE,
					'options'     => array(
						'btn-sm' => array(
							'title' => esc_html__( 'Small', 'alpha-core' ),
							'icon'  => 'alpha-size-sm alpha-choose-type',
						),
						'btn-md' => array(
							'title' => esc_html__( 'Medium', 'alpha-core' ),
							'icon'  => 'alpha-size-md alpha-choose-type',
						),
						'btn-lg' => array(
							'title' => esc_html__( 'Large', 'alpha-core' ),
							'icon'  => 'alpha-size-lg alpha-choose-type',
						),
						'btn-xl' => array(
							'title' => esc_html__( 'Extra Large', 'alpha-core' ),
							'icon'  => 'alpha-size-xl alpha-choose-type',
						),
					),
				)
			);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_filter_general',
			array(
				'label' => esc_html__( 'General', 'alpha-core' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

			$this->add_control(
				'filter_br',
				array(
					'label'       => esc_html__( 'Border Radius', 'alpha-core' ),
					'type'        => Controls_Manager::SLIDER,
					'description' => esc_html__( 'Set the filter boxes border radius.', 'alpha-core' ),
					'size_units'  => array(
						'px',
						'%',
					),
					'range'       => array(
						'step' => 1,
						'min'  => 0,
						'max'  => 100,
					),
					'selectors'   => array(
						'.elementor-element-{{ID}} .alpha-filters' => '--alpha-filter-border-radius: {{SIZE}}{{UNIT}}',
					),
				)
			);

			$this->add_control(
				'filter_gap',
				array(
					'type'        => Controls_Manager::SLIDER,
					'size_units'  => array(
						'px',
						'rem',
						'%',
					),
					'label'       => esc_html__( 'Gap', 'alpha-core' ),
					'description' => esc_html__( 'Set gap space between filter boxes.', 'alpha-core' ),
					'selectors'   => array(
						'.elementor-element-{{ID}} .align-left > *' => "margin-{$right}: {{SIZE}}{{UNIT}};",
						'.elementor-element-{{ID}} .align-center > *' => "margin-{$left}: calc( {{SIZE}}{{UNIT}} / 2 ); margin-{$right}: calc( {{SIZE}}{{UNIT}} / 2 );",
						'.elementor-element-{{ID}} .align-right > *' => "margin-{$left}: {{SIZE}}{{UNIT}};",
					),
				)
			);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_filter_style',
			array(
				'label' => esc_html__( 'Filter Boxes', 'alpha-core' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

			$this->add_group_control(
				Group_Control_Typography::get_type(),
				array(
					'name'     => 'filter_typography',
					'label'    => esc_html__( 'Typography', 'alpha-core' ),
					'selector' => '{{WRAPPER}} .alpha-filter',
				)
			);

			$this->add_responsive_control(
				'filter_responsive_padding',
				array(
					'label'       => esc_html__( 'Padding', 'alpha-core' ),
					'description' => esc_html__( 'Control the filter box padding.', 'alpha-core' ),
					'type'        => Controls_Manager::DIMENSIONS,
					'size_units'  => array(
						'px',
						'%',
						'em',
					),
					'selectors'   => array(
						'{{WRAPPER}} .select-ul-toggle' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					),
				)
			);

		$this->end_controls_section();
	}

	protected function render() {
		$atts = $this->get_settings_for_display();
		require alpha_core_framework_path( ALPHA_CORE_FRAMEWORK_PATH . '/widgets/filter/render-filter.php' );
	}
}
